
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 30 10:17:10 2008
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief A brief description of this module 
   *
   *  \version CRYS_SEP_ATP_AES_Acceptance.c#1:csrc:1
   *  \author 
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include "CRYS.h"
#include "CRYS_SEP_ATP_Defines.h"
#include "DX_VOS_Mem.h"


/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/
CRYSError_t      TST_ATP_Error;

CRYS_AESUserContext_t	 ContextID;
/************* Private function prototype ***********************/
/************************ Public Functions ******************************/

/********************************************************************************
* Function Name:
*  SEPQA_ATP_CheckErrorReturn
*
* Inputs:
*  DxUint32_t	TST_ErrorRet - The return value of the checked function
*  char			TST_FuncName[TST_FUNC_STRING_LEN] - The checked function name
*  char			FailureFunctionNameOut_ptr[TST_FUNC_STRING_LEN] - In case of function
*                                 failure will return the function name that fails
*
* Outputs:
*  DxUint8_t - Function error return
*
* Description:
* This function Is a utility function to save up some space
* this function returns the appropriate printing for a checked function
* If the operation has succeeded of failed the appropriate printing and return value
*
********************************************************************************/
CRYSError_t SEPQA_ATP_CheckErrorReturn(CRYSError_t  TST_ErrorRet,
									   char         TST_FuncName[TST_FUNC_STRING_LEN],
									   char         FailureFunctionNameOut_ptr[TST_FUNC_STRING_LEN])
{
	if (TST_ErrorRet != CRYS_OK)
	{
		/*Copying the input buffer onto output buffer not to overwrite test data*/
		DX_VOS_FastMemCpy(FailureFunctionNameOut_ptr,
						  TST_FuncName,
						  TST_FUNC_STRING_LEN);
				  
		return TST_ErrorRet;
	}
	else
	{
		return TESTPASS;
	}
}


/********************************************************************************
* Function Name:
*  SEPQA_AES_NotIntegrated
*
* Inputs:
* IVCounter_ptr - this parameter is the buffer of the IV or counters on mode CTR.
*                          In ECB mode this parameter has no use.
*                          In CBC and MAC modes this parameter should containe the IV values.
*                          in CTR mode this value shuld contain the init counter.
*
* Key_ptr - a pointer to the users key buffer.
* KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
* EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
*                           Decrypt operation [1].
* OperationMode - The operation mode : ECB , CBC , MAC or CTR.
* is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
* DataIn_ptr - The pointer to the buffer of the input data to the AES. 
*                   The pointer's value does not need to be word-aligned.
* DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  DxUint8_t - Function error return
*
* Description:
* This function is qa external application on the SEP, 
* that executes AES not integrated test
* Algorithm:     
* 1. Allocate memory buffers 
* 2. Calling the AES init function
* 3. Call the CRYS_AES_Block - This function is called N times 
*                              (N = number of blocks in data each block is 16byte) 
* 4. Calling the Free function
* 5. Compare the outputted result to the expected one  
* 6. Free allocated buffers
********************************************************************************/
CRYSError_t SEPQA_AES_NotIntegrated(CRYS_AES_IvCounter_t       IVCounter_ptr,       
									CRYS_AES_Key_t             Key_ptr,
									CRYS_AES_KeySize_t         KeySize,            
									CRYS_AES_EncryptMode_t     EncryptDecryptFlag, 
									CRYS_AES_OperationMode_t   OperationMode ,             
									DxUint8_t                  *DataIn_ptr,        
									DxUint32_t                 DataInSize,
									DxUint8_t                  *ExpectedData_ptr,        
									DxUint32_t                 DataExpSize,
									DxUint32_t				   OutputLocation,
									char                      *FailureFunctionNameOut_ptr)
{
	/********* TST Variables Definitions ************/
	CRYSError_t              TST_TestStatus=0;
	DxUint32_t               Block_index = 0;
	DxUint32_t               NumOfBlocks = 0;
	DxUint8_t				*pDataIn;
	DxUint8_t				*pDataOut;

	/* data expected size = data in size */
	DataExpSize = DataInSize;


	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(DataInSize);

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);
		
	/* 2. calling the AES init function */ 
	/************************************/ 
	TST_ATP_Error = CRYS_AES_Init(&ContextID,
								  IVCounter_ptr, 
							      Key_ptr,
							      KeySize,
							      EncryptDecryptFlag,  
							      OperationMode);

	/* check error return */
	/**********************/
	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
											    "CRYS_AES_Init",
												FailureFunctionNameOut_ptr);
		

	if (TST_TestStatus != TESTPASS) 
		goto End;


	/* 3. calling the AES Block function */
	/*************************************/
	NumOfBlocks = DataInSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;


	if (OutputLocation == 0)/* input buffer != output buffer */
	{
		for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*AES block loop*/ 
		{   
			if (EncryptDecryptFlag == CRYS_AES_Encrypt)
			{ 
			
				TST_ATP_Error = CRYS_AES_Block( &ContextID,     
												pDataIn+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
												TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
												pDataOut+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); 


				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
															"CRYS_AES_Block - Encrypt",
															FailureFunctionNameOut_ptr);				
			}
			else
			{                           
				TST_ATP_Error = CRYS_AES_Block( &ContextID,     
												ExpectedData_ptr+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),         
												TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
												pDataOut+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index));   

				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
															"CRYS_AES_Block - Decrypt",
															FailureFunctionNameOut_ptr);								
			}   

			if (TST_TestStatus != TESTPASS) 
				goto End;
		}/*End AES block loop*/ 
	} /* if (OutputLocation == 0) */
	else /* input buffer = output buffer */ 
	{
		for (Block_index =0 ; Block_index < NumOfBlocks  ; Block_index++)
		{
			if (EncryptDecryptFlag == CRYS_AES_Encrypt)
			{
				/*Copying the input buffer onto output buffer not to overwrite test data*/
				DX_VOS_FastMemCpy(pDataOut + (TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
								  pDataIn+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
									TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
			}
			else
			{
				/*Copying the expected output buffer onto output buffer not to overwrite test data*/
				DX_VOS_FastMemCpy(pDataOut+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
								  ExpectedData_ptr+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index), 
								  TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
			}
		} /*End AES block loop*/            

		for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
		{   
			TST_ATP_Error = CRYS_AES_Block(&ContextID,           
										   pDataOut+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
										   TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
										   pDataOut+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/        

			if (EncryptDecryptFlag == CRYS_AES_Encrypt)
				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
															"CRYS_AES_Block - Encrypt",
															FailureFunctionNameOut_ptr);								
			else
				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
															"CRYS_AES_Block - Decrypt",
															FailureFunctionNameOut_ptr);								
				
			if (TST_TestStatus != TESTPASS) 
				goto End;

		}
	}   


	/* 4. calling the Free function */ 
	/********************************/     
	TST_ATP_Error = CRYS_AES_Free( &ContextID );

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_AES_Free",
												FailureFunctionNameOut_ptr);								

	if (TST_TestStatus != TESTPASS) 
		goto End;



	/* 5. Compare the result to expected result */ 
	/********************************************/ 
	if (EncryptDecryptFlag == CRYS_AES_Encrypt)  
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(	pDataOut,
													ExpectedData_ptr, 
													DataInSize);
	}
	else
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut ,
													DataIn_ptr,
													DataInSize);                
	}

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"DX_VOS_MemCmp",
												FailureFunctionNameOut_ptr);								

End:
	/* 6. Free allocated buffers */
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;

} /* End of first if statement*/

/********************************************************************************
* Function Name:
*  SEPQA_AES_Integrated
*
* Inputs:
* IVCounter_ptr - this parameter is the buffer of the IV or counters on mode CTR.
*                          In ECB mode this parameter has no use.
*                          In CBC and MAC modes this parameter should containe the IV values.
*                          in CTR mode this value shuld contain the init counter.
*
* Key_ptr - a pointer to the users key buffer.
* KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
* EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
*                           Decrypt operation [1].
* OperationMode - The operation mode : ECB , CBC , MAC or CTR.
* is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
* DataIn_ptr - The pointer to the buffer of the input data to the AES. 
*                   The pointer's value does not need to be word-aligned.
* DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  DxUint8_t - Function error return
*
* Description:
* This function is qa external application on the SEP, 
* that executes AES integrated test
* Algorithm:     
* 1. Allocate memory buffers
* 2. Calling the AES integrated function
* 3. Compare the result to expected result
* 4. Free allocated buffers
********************************************************************************/
CRYSError_t SEPQA_AES_Integrated(
								CRYS_AES_IvCounter_t       IVCounter_ptr,       
								CRYS_AES_Key_t             Key_ptr,
								CRYS_AES_KeySize_t         KeySize,            
								CRYS_AES_EncryptMode_t     EncryptDecryptFlag, 
								CRYS_AES_OperationMode_t   OperationMode ,             
								DxUint8_t                  *DataIn_ptr,        
								DxUint32_t                 DataInSize,
								DxUint8_t                  *ExpectedDataOut_ptr,        
								DxUint32_t                 DataExpSize,
								int						   OutputLocation,
								char                      *FailureFunctionNameOut_ptr)
{
	/********* TST Variables Definitions ************/
	CRYSError_t              TST_TestStatus = TESTPASS;
	DxUint8_t				*pDataIn = DX_NULL;
	DxUint8_t				*pDataOut = DX_NULL;

	/* data expected size = data in size */
	DataExpSize = DataInSize;
	
	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);			
	pDataOut = DX_VOS_MemMalloc(DataInSize);

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);
		
	/* 2. calling the AES integrated function */
	/******************************************/ 
	if (OutputLocation == 0)/* input buffer != output buffer */
	{
		if (EncryptDecryptFlag ==CRYS_AES_Encrypt) 
		{
			TST_ATP_Error = CRYS_AES(IVCounter_ptr,
				Key_ptr,
				KeySize,
				EncryptDecryptFlag,
				OperationMode,
				pDataIn, 
				DataInSize, 
				pDataOut);
				
		}
		else
		{
			TST_ATP_Error = CRYS_AES(IVCounter_ptr,
				Key_ptr,
				KeySize,
				EncryptDecryptFlag,
				OperationMode,
				ExpectedDataOut_ptr, 
				DataInSize, 
				pDataOut); 
		}
	}
	else /* input buffer = output buffer */ 
	{
		if (EncryptDecryptFlag == CRYS_AES_Encrypt)
		{
			/*Copying the input buffer onto output buffer not to overwrite test data*/
			DX_VOS_FastMemCpy(pDataOut,
							  pDataIn,
							  DataInSize);  
		}
		else
		{
			/*Copying the input buffer onto output buffer not to overwrite test data*/
			DX_VOS_FastMemCpy(pDataOut,
							  ExpectedDataOut_ptr, 
							  DataInSize);  
		}
		
		TST_ATP_Error = CRYS_AES(IVCounter_ptr,
								Key_ptr,
								KeySize,
								EncryptDecryptFlag,
								OperationMode,
								pDataOut, 
								DataInSize, 
								pDataOut);
		
	} 

	if (EncryptDecryptFlag == CRYS_AES_Encrypt)
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_AES Encrypt",
													FailureFunctionNameOut_ptr);										
	else
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_AES Decrypt",
													FailureFunctionNameOut_ptr);								

	if (TST_TestStatus != TESTPASS) 
		goto End;
 

	/* 3. Compare the result to expected result */ 
	/*****************************************/                                                             
	if (EncryptDecryptFlag == CRYS_AES_Encrypt)
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(	pDataOut, ExpectedDataOut_ptr, DataInSize);
	}
	else
	{
		TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(	pDataOut, pDataIn, DataInSize);                
	}               
	
	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"DX_VOS_MemCmp",
												FailureFunctionNameOut_ptr);								
End:
	/* 4. Free allocated buffers */
	/*************************/
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;
}



/********************************************************************************
* Function Name:
*  SEPQA_AES_MAC_NON_Integrated
*
* Inputs:
* IVCounter_ptr - this parameter is the buffer of the IV or counters on mode CTR.
*                          In ECB mode this parameter has no use.
*                          In CBC and MAC modes this parameter should containe the IV values.
*                          in CTR mode this value shuld contain the init counter.
*
* Key_ptr - a pointer to the users key buffer.
* KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
* EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
*                           Decrypt operation [1].
* OperationMode - The operation mode : ECB , CBC , MAC or CTR.
* is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
* DataIn_ptr - The pointer to the buffer of the input data to the AES. 
*                   The pointer's value does not need to be word-aligned.
* DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  DxUint8_t - Function error return
*
* Description:
* This function is qa external application on the SEP, 
* that executes AES not integrated test
* Algorithm:     
* 1. Allocate memory buffers 
* 2. Calling the AES init function
* 3. Call the CRYS_AES_Block - This function is called N times 
*                              (N = number of blocks in data each block is 16byte) 
* 4. Calling the Free function
* 5. Compare the outputted result to the expected one  
* 6. Free allocated buffers
********************************************************************************/
CRYSError_t SEPQA_AES_MAC_NON_Integrated(CRYS_AES_IvCounter_t       IVCounter_ptr,       
										 CRYS_AES_Key_t             Key_ptr,
										 CRYS_AES_KeySize_t         KeySize,            
										 CRYS_AES_EncryptMode_t     EncryptDecryptFlag, 
										 CRYS_AES_OperationMode_t   OperationMode ,             
										 DxUint8_t                  *DataIn_ptr,        
										 DxUint32_t                 DataInSize,
										 DxUint8_t*                 ExpectedDataOut_ptr,        
										 int						OutputLocation,
										 char                      *FailureFunctionNameOut_ptr)
{
	/********* TST Variables Definitions ************/
	CRYSError_t              TST_TestStatus=0;
	DxUint32_t               Block_index = 0;
	DxUint32_t               NumOfBlocks = 0;
	DxUint32_t               Index = 0;
	DxUint8_t				*pDataIn = DX_NULL;
	DxUint8_t				*pDataOut = DX_NULL;
	
	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(DataInSize);

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{

		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	/* 2. calling the AES init function */ 
	/************************************/ 
	TST_ATP_Error = CRYS_AES_Init(&ContextID,
		IVCounter_ptr, 
		Key_ptr,
		KeySize,
		EncryptDecryptFlag,  
		OperationMode);

	/* check error return */
	/**********************/
	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_AES_Init",
		FailureFunctionNameOut_ptr);

	if (TST_TestStatus != TESTPASS) 
		goto End;


	/* 3. calling the AES Block function */
	/*************************************/
	NumOfBlocks = DataInSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;



	if (OutputLocation == 0)/* input buffer != output buffer */
	{
		for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*AES block loop*/ 
		{   
			if (EncryptDecryptFlag == CRYS_AES_Encrypt)
			{ 
				TST_ATP_Error = CRYS_AES_Block( &ContextID,     
					pDataIn+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
					TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
					pDataOut+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); 

				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
					"CRYS_AES_Block - Encrypt",
					FailureFunctionNameOut_ptr);				
			}
			else
			{                           
				TST_ATP_Error = CRYS_AES_Block( &ContextID,     
					ExpectedDataOut_ptr+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),         
					TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
					pDataOut+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index));   

				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
					"CRYS_AES_Block - Decrypt",
					FailureFunctionNameOut_ptr);								
			}   

			if (TST_TestStatus != TESTPASS) 
				goto End;


		}/*End AES block loop*/ 
	}
	else /* input buffer = output buffer */ 
	{
		for (Block_index =0 ; Block_index < NumOfBlocks  ; Block_index++)
		{
			if (EncryptDecryptFlag == CRYS_AES_Encrypt)
			{
				/*Copying the input buffer onto output buffer not to overwrite test data*/
				DX_VOS_FastMemCpy(pDataOut + (TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
					pDataIn+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
					TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
			}
			else
			{
				/*Copying the expected output buffer onto output buffer not to overwrite test data*/
				DX_VOS_FastMemCpy(pDataOut+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
					ExpectedDataOut_ptr+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index), 
					TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
			}
		}           

		for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
		{   
			TST_ATP_Error = CRYS_AES_Block(&ContextID,           
				pDataOut+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
				TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
				pDataOut+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/        

			if (EncryptDecryptFlag == CRYS_AES_Encrypt)
				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
													"CRYS_AES_Block - Encrypt",
													FailureFunctionNameOut_ptr);								
			else
				TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_AES_Block - Decrypt",
				FailureFunctionNameOut_ptr);								

			if (TST_TestStatus != TESTPASS) 
				goto End;

		}
	}    

	/* 4. calling the Free function */ 
	/********************************/        
	TST_ATP_Error = CRYS_AES_Free( &ContextID );

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
		"CRYS_AES_Free",
		FailureFunctionNameOut_ptr);								

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 5. Compare the result to expected result */ 
	/********************************************/
	for (Index =0; Index<TST_CBC_MAC_SIZE ;Index++) 
	{
		if (ExpectedDataOut_ptr[Index] !=
			pDataOut[((NumOfBlocks-1)*TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS)+Index])
			TST_ATP_Error = TESTFAIL;
	}

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"DX_VOS_MemCmp",
												FailureFunctionNameOut_ptr);								
End:
	/* 6. Free allocated buffers */
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;

}

/********************************************************************************
* Function Name:
*  SEPQA_AES_XCBC_CMACNonIntegrated
*
* Inputs:
*
* Key_ptr - a pointer to the users key buffer.
* KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
* EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
*                           Decrypt operation [1].
* OperationMode - The operation mode : ECB , CBC , MAC or CTR.
* is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
* DataIn_ptr - The pointer to the buffer of the input data to the AES. 
*                   The pointer's value does not need to be word-aligned.
* DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  DxUint8_t - Function error return
*
* Description:
* This function is qa external application on the SEP 
* This function Encrypts and Decrypts known vectors using AES new modes
* XCBC and CMAC
* AES modes                   - MAC   
* Vector sizes used (in bytes) - 1536                     
* Key Size used (in bits)     -  256 CMAC only                              
* This also outputs the results to an output buffer or the same input buffer
* Algorithm:     
* 1. Allocate memory buffers 
* 2. Call the CRYS_AES_Init_XCBCandCMAC function to start the HASH process
* 3. Call the CRYS_AES_Block - This function is called N times 
*                              (N = number of blocks in data each block is 16byte) 
* 4. Call the CRYS_AES_XCBCandCMAC_Finish to end the encryption/decryption process 
* 5. Compare the outputted result to the expected one                                                   
*
******************************************************************************/

CRYSError_t SEPQA_AES_XCBC_CMACNonIntegrated(CRYS_AES_Key_t             Key_ptr,
											 CRYS_AES_KeySize_t         KeySize, 

											 CRYS_AES_OperationMode_t   OperationMode ,      
											 DxUint8_t                  *DataIn_ptr,        
											 DxUint32_t                 DataInSize,
											 DxUint8_t*                 ExpectedDataOut_ptr,        
											 int						OutputLocation,
											 char                      *FailureFunctionNameOut_ptr)
{
/********* TST Variables Definitions ************/
	DxUint32_t              Block_index = 0;
	DxUint32_t				NumOfBlocks = 0;
	CRYSError_t             TST_TestStatus=0;
	DxUint32_t              Index = 0;
	DxUint8_t				*pDataIn = DX_NULL;
	DxUint8_t				*pDataOut = DX_NULL;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(DataInSize);

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{
		/* check error return */
		/**********************/
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut_ptr);

		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	TST_TestStatus = TESTPASS;

	/* 2. calling the AES init function */ 
	/************************************/ 
	if (TST_USE_SECRETKEY_TESTS) 
	{
		TST_ATP_Error = CRYS_AES_Init_XCBCandCMAC_SecretKey(&ContextID, OperationMode);
	}
	else
	{
		TST_ATP_Error = CRYS_AES_Init_XCBCandCMAC(&ContextID, Key_ptr, KeySize, OperationMode);
	}


	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_AES_Init_XCBCandCMAC",
												FailureFunctionNameOut_ptr);								

	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 3. calling the AES Block function */
	/*************************************/
	NumOfBlocks = DataInSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;

	if (OutputLocation == 0) 
	{
		for (Block_index = 0 ; Block_index < NumOfBlocks    ; Block_index++)
		{   
			TST_ATP_Error = CRYS_AES_Block( &ContextID,  pDataIn + (TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
				TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
				pDataOut + (TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/ 

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_AES_Block",
				FailureFunctionNameOut_ptr);								

			if (TST_TestStatus != TESTPASS) 
				goto End;         
		}
	}
	else /* input buffer = output buffer */ 
	{

		DX_VOS_FastMemCpy(pDataOut, pDataIn, DataInSize);

		for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
		{   
			TST_ATP_Error = CRYS_AES_Block(&ContextID, pDataOut + (TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),        
				TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t), 
				pDataOut + (TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index)); /*(DxUint8_t *)TST_output_data);*/

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
				"CRYS_AES_Block",
				FailureFunctionNameOut_ptr);								

			if (TST_TestStatus != TESTPASS) 
				goto End;  
		}

	}    

	/* 4. calling the Finish function */ 
	/********************************/
	TST_ATP_Error = CRYS_AES_XCBCandCMAC_Finish(&ContextID, pDataOut );

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"CRYS_AES_XCBCandCMAC_Finish",
												FailureFunctionNameOut_ptr);								

	if (TST_TestStatus != TESTPASS) 
		goto End;     


	/* 5. Compare the result to expected result */ 
	/********************************************/
	for (Index =0; Index < TST_CBC_MAC_SIZE; Index++) 
	{
		if (ExpectedDataOut_ptr[Index] != pDataOut[Index])
			TST_ATP_Error = TESTFAIL;
	}
	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"Mem_Compare",
												FailureFunctionNameOut_ptr);								

	if (TST_TestStatus != TESTPASS) 
		goto End; 

End:
	/* 6. Free allocated buffers */
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;
}


/*****************************************************************************
* Function Name:                                                           
*  SEPQA_AES_XCBC_CMACIntegrated      
*  
* Inputs:
* Key_ptr - a pointer to the users key buffer.
* KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
* EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
*                           Decrypt operation [1].
* OperationMode - The operation mode : ECB , CBC , MAC or CTR.
* is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
* DataIn_ptr - The pointer to the buffer of the input data to the AES. 
*                   The pointer's value does not need to be word-aligned.
* DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for AES non integrated API's                         
*  This function Encrypts and Decrypts known vectors using AES new modes
*  XCBC and CMAC
*  AES modes                   - XCBX/CMAC   
*  Vector sizes used (in bytes) - 1536                     
*  Key Size used (in bits)     -  256 CMAC only/Secret key 
*  
*  This also outputs the results to an output buffer or the same input buffer
* Algorithm:       
* 1. Call the CRYS_AES_XCBCandCMAC or CRYS_AES_XCBCandCMAC_SecretKey function to start the HASH process
* 2. Compare the outputted result to the expected one                                                   
*
******************************************************************************/

CRYSError_t SEPQA_AES_XCBC_CMACIntegrated(CRYS_AES_Key_t                Key_ptr,
										     CRYS_AES_KeySize_t         KeySize,            
										     CRYS_AES_OperationMode_t   OperationMode ,             
										     DxUint8_t                  *DataIn_ptr,        
										     DxUint32_t                 DataInSize,
										     DxUint8_t*                 ExpectedDataOut_ptr,        
										     int						OutputLocation,
										     char                       *FailureFunctionNameOut_ptr )
{
	CRYSError_t             TST_TestStatus = 0;
	DxUint32_t              Index = 0;
	DxUint8_t				*pDataIn = DX_NULL;
	DxUint8_t				*pDataOut = DX_NULL;

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(DataInSize);

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{
		/* check error return */
		/**********************/
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut_ptr);
		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	TST_TestStatus = TESTPASS;

	/* 2. calling the AES init function */ 
	/************************************/ 
	if (OutputLocation == 0) 
	{
		if (TST_USE_SECRETKEY_TESTS) 
		{
			TST_ATP_Error = CRYS_AES_XCBCandCMAC_SecretKey(OperationMode, pDataIn,
				DataInSize, pDataOut);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error, 
														"CRYS_AES_XCBCandCMAC_SecretKey",
														FailureFunctionNameOut_ptr);
			if (TST_TestStatus != TESTPASS) 
				goto End;

		}
		else
		{
			TST_ATP_Error = CRYS_AES_XCBCandCMAC(Key_ptr, KeySize, OperationMode,
				pDataIn, DataInSize,pDataOut);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error, 
														"CRYS_AES_XCBCandCMAC",
														FailureFunctionNameOut_ptr);
			if (TST_TestStatus != TESTPASS) 
				goto End;

		} 
	}
	else /* input buffer = output buffer */ 
	{              

		DX_VOS_FastMemCpy(pDataOut, pDataIn, DataInSize); 

		if (TST_USE_SECRETKEY_TESTS) 
		{
			TST_ATP_Error = CRYS_AES_XCBCandCMAC_SecretKey(OperationMode, pDataOut,	DataInSize,	pDataOut);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error, 
														"CRYS_AES_XCBCandCMAC_SecretKey",
														FailureFunctionNameOut_ptr);
			if (TST_TestStatus != TESTPASS) 
				goto End;
		}
		else
		{
			TST_ATP_Error = CRYS_AES_XCBCandCMAC(Key_ptr, KeySize, OperationMode,
				pDataOut, DataInSize, pDataOut);

			TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error, 
														"CRYS_AES_XCBCandCMAC",
														FailureFunctionNameOut_ptr);
			if (TST_TestStatus != TESTPASS) 
				goto End;

		} 
	}        

	/* 3. Compare the result to expected result */ 
	/********************************************/
	for (Index =0; Index<TST_CBC_MAC_SIZE ;Index++) 
	{
		if (ExpectedDataOut_ptr[Index] != pDataOut[Index])
			TST_ATP_Error = TESTFAIL;
	}

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"DX_VOS_MemCmp",
												FailureFunctionNameOut_ptr);	
End:
	/* 4. Free allocated buffers */
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;
}

/*****************************************************************************
* Function Name:                                                           
*  SEPQA_AES_MAC_Integrated      
*  
*  Inputs:
* Key_ptr - a pointer to the users key buffer.
* KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
* EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
*                           Decrypt operation [1].
* OperationMode - The operation mode : ECB , CBC , MAC or CTR.
* is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
* DataIn_ptr - The pointer to the buffer of the input data to the AES. 
*                   The pointer's value does not need to be word-aligned.
* DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for AES non integrated API's                         
*  This function Encrypts and Decrypts known vectors using AES
*  AES modes                   - MAC/XCBC/CMAC   
*  Vector sizes used (in bytes) - 16/32/512/1536                           
*  Key Size used (in bits)     - 128/192(MAC only)/256(MAC only)
*  This also outputs the results to an output buffer or the same input buffer
* Algorithm:       
* 1. Call the CRYS_AES_Init function to start the encryption/decryption process
* 2. Compare the outputted result to the expected one                                                   
*
******************************************************************************/
CRYSError_t SEPQA_AES_MAC_Integrated(CRYS_AES_IvCounter_t       IVCounter_ptr,       
									 CRYS_AES_Key_t             Key_ptr,
									 CRYS_AES_KeySize_t         KeySize,            
									 CRYS_AES_EncryptMode_t     EncryptDecryptFlag, 
									 CRYS_AES_OperationMode_t   OperationMode ,             
									 DxUint8_t                  *DataIn_ptr,        
									 DxUint32_t                 DataInSize,
									 DxUint8_t*                 ExpectedDataOut_ptr,        
									 int						OutputLocation,
									 char						*FailureFunctionNameOut_ptr )
{
	CRYSError_t             TST_TestStatus=0;
	DxUint32                Block_index = 0; 
	DxUint32				NumOfBlocks = 0;
	DxUint32				Index = 0;
	DxUint8_t				*pDataIn = DX_NULL;
	DxUint8_t				*pDataOut = DX_NULL;
	

	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(DataInSize);

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL))
	{
		/* check error return */
		/**********************/
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
													"DX_VOS_MemMalloc",
													FailureFunctionNameOut_ptr);
		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	TST_TestStatus = TESTPASS;

	NumOfBlocks = DataInSize/TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS;

	/* 2. calling the AES integrated function */ 
	/************************************/ 
	if (OutputLocation == 0 )/* output buffer != input buffer    */
	{   
		TST_ATP_Error = CRYS_AES(IVCounter_ptr, Key_ptr, KeySize,
			EncryptDecryptFlag, OperationMode, 
			pDataIn, DataInSize, pDataOut);                                
	}
	else /* input buffer = output buffer */ 
	{
		for (Block_index =0 ; Block_index < NumOfBlocks ; Block_index++)
		{   
			/*Copying the input buffer onto output buffer not to overwrite test data*/
			DX_VOS_FastMemCpy(pDataOut+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
				pDataIn+(TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS*Block_index),
				TST_MAX_SINGLE_AES_BLOCK_SIZE_IN_BYTS);  
		}       

		TST_ATP_Error = CRYS_AES(IVCounter_ptr, Key_ptr, KeySize,
			EncryptDecryptFlag, OperationMode,
			pDataOut, DataInSize, pDataOut);           
	} 

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error, 
												"CRYS_AES",
												FailureFunctionNameOut_ptr);
	if (TST_TestStatus != TESTPASS) 
		goto End;


	/* 3. Compare the result to expected result */ 
	/********************************************/
	for (Index =0; Index<TST_CBC_MAC_SIZE ;Index++) 
	{
		if (ExpectedDataOut_ptr[Index] != pDataOut[Index])
			TST_ATP_Error = TESTFAIL;
	}

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error,
												"DX_VOS_MemCmp",
												FailureFunctionNameOut_ptr);	

End:
	/* 4. Free allocated buffers */
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);

	return TST_TestStatus;
}


/*****************************************************************************
* Function Name:                                                           
*  SEPQA_AES_KeyWrapUnwarp      
*  
*  Inputs:
* Key_ptr - a pointer to the users key buffer.
* KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
* is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
* DataIn_ptr - The pointer to the buffer of the input data to the AES. 
*                   The pointer's value does not need to be word-aligned.
* DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* WrapdDataSize - The size of the expected data after wrap operation
* 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for AES non integrated API's                         
*  This function Encrypts and Decrypts known vectors using AES
*  AES modes                   - MAC/XCBC/CMAC   
*  Vector sizes used (in bytes) - 1536                           
*  Key Size used (in bits)     -  256(MAC only)
*  This also outputs the results to an output buffer or the same input buffer
* Algorithm:       
* 1. Call the CRYS_Wrap and Unwrap function to use the AES wrap/unwrap process
* 2. Compare the outputted result to the expected one                                                  
*
******************************************************************************/
CRYSError_t SEPQA_AES_KeyWrapUnwarp(CRYS_AES_Key_t             Key_ptr,
									CRYS_AES_KeySize_t         KeySize,                  				       
									DxUint8_t                  *DataIn_ptr,        
									DxUint32_t                 DataInSize,
									DxUint8_t*                 ExpectedDataOut_ptr,        
									DxUint32_t                 WrapdDataSize,
									char					   *FailureFunctionNameOut_ptr )
{
	CRYSError_t             TST_TestStatus=0;
	DxUint32_t              TST_WrapdDataActualSize = WrapdDataSize; 
	DxUint32_t              TST_OutputDataActualSize = 0;
	DxUint8_t               is_secret_key = DX_FALSE;
	DxUint8_t				*pDataIn = DX_NULL;
	DxUint8_t				*pDataOut = DX_NULL;
	DxUint8_t				*pWrapdata = DX_NULL;


	/* 1. Allocate memory buffers */
	/******************************/
	pDataIn = DX_VOS_MemMalloc(DataInSize);
	pDataOut = DX_VOS_MemMalloc(DataInSize);
	pWrapdata = DX_VOS_MemMalloc(WrapdDataSize);

	if ((pDataIn == DX_NULL) ||
		(pDataOut == DX_NULL)||
		(pWrapdata == DX_NULL))
	{
		/* check error return */
		/**********************/
		TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TESTFAIL,
			"DX_VOS_MemMalloc",
			FailureFunctionNameOut_ptr);
		goto End;                                                                       
	}

	DX_VOS_FastMemCpy(pDataIn, DataIn_ptr, DataInSize);

	TST_TestStatus = TESTPASS;
	/* 2. calling the AES Wrap function */ 
	/************************************/ 
	if (TST_USE_SECRETKEY_TESTS) 
		is_secret_key=DX_TRUE;

	TST_ATP_Error = CRYS_AES_Wrap(pDataIn, DataInSize, Key_ptr, KeySize,
		is_secret_key, pWrapdata, &TST_WrapdDataActualSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error, 
												"CRYS_AES_Wrap",
												FailureFunctionNameOut_ptr);
	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 3. Compare the result to expected result */ 
	/********************************************/                                                            
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pWrapdata, ExpectedDataOut_ptr, TST_WrapdDataActualSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error, 
												"DX_VOS_MemCmp",
												FailureFunctionNameOut_ptr);
	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* No to effect data in the struct*/
	TST_OutputDataActualSize = DataInSize;

	/* 4. calling the AES Unwrap function */ 
	/************************************/
	TST_ATP_Error = CRYS_AES_Unwrap(pWrapdata, WrapdDataSize, 
		Key_ptr, KeySize, is_secret_key,
		pDataOut, &TST_OutputDataActualSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error, 
												"CRYS_AES_Unwrap",
												FailureFunctionNameOut_ptr);
	if (TST_TestStatus != TESTPASS) 
		goto End;

	/* 5. Compare the result to expected result */ 
	/********************************************/                                                            
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(pDataOut, pDataIn, TST_OutputDataActualSize);

	TST_TestStatus = SEPQA_ATP_CheckErrorReturn(TST_ATP_Error, 
												"DX_VOS_MemCmp",
												FailureFunctionNameOut_ptr);

End:
	/* 6. Free allocated buffers */
	DX_VOS_MemFree(pDataIn);
	DX_VOS_MemFree(pDataOut);
	DX_VOS_MemFree(pWrapdata);

	return TST_TestStatus;

}
